<?php

namespace App\Http\Controllers\Transaction;

use App\Http\Controllers\Controller;
use App\Models\Sale;
use App\Models\SaleDetail;
use App\Models\Product;
use App\Models\ProductStock;
use App\Models\ProductPrice;
use App\Models\Customer;
use App\Models\Branch;
use App\Models\StockMovement;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Barryvdh\DomPDF\Facade\Pdf;
use BaconQrCode\Renderer\ImageRenderer;
use BaconQrCode\Renderer\Image\SvgImageBackEnd;
use BaconQrCode\Renderer\RendererStyle\RendererStyle;
use BaconQrCode\Writer;
use BaconQrCode\Common\ErrorCorrectionLevel;
use BaconQrCode\Encoder\Encoder;

class SaleController extends Controller
{
    public function __construct()
    {
        $this->middleware(['permission:sales.index'])->only('index', 'show');
        $this->middleware(['permission:sales.create'])->only(['create', 'save']);
        $this->middleware(['permission:sales.edit'])->only(['edit', 'update']);
        $this->middleware(['permission:sales.delete'])->only('delete');
    }

    /**
     * Display a listing of sales
     */
    public function index()
    {
        $title = "sale";
        $sales = Sale::with(['customer', 'branch', 'cashier'])
            ->orderBy('id', 'desc')
            ->get();

        return view('transaction.sale.index', compact('title', 'sales'));
    }

    /**
     * Show the form for creating a new sale
     */
    public function create()
    {
        $title = "sale";
        $customers = Customer::orderBy('id', 'desc')->get();
        $branches = Branch::orderBy('id', 'desc')->get();
        $products = Product::with(['brand', 'category', 'productStocks', 'productPrices'])->get();

        // Generate sale number
        $saleNumber = $this->generateSaleNumber();

        return view('transaction.sale.create', compact('title', 'customers', 'branches', 'products', 'saleNumber'));
    }

    /**
     * Get product stock and price for a specific branch
     */
    public function getProductInfo(Request $request)
    {
        $productId = $request->product_id;
        $branchId = $request->branch_id;

        $stock = ProductStock::where('product_id', $productId)
            ->where('branch_id', $branchId)
            ->first();

        $price = ProductPrice::where('product_id', $productId)
            ->where('branch_id', $branchId)
            ->orderBy('effective_date', 'desc')
            ->first();

        return response()->json([
            'stock' => $stock ? $stock->quantity : 0,
            'purchase_price' => $price ? $price->purchase_price : 0,
            'selling_price' => $price ? $price->selling_price : 0,
        ]);
    }

    /**
     * Store a newly created sale
     */
    public function save(Request $request)
    {
        $this->validate($request, [
            'branch_id' => 'required|exists:branches,id',
            'customer_id' => 'nullable|exists:customers,id',
            'sale_date' => 'required|date',
            'payment_method' => 'required|in:cash,transfer,credit',
            'details' => 'required|array|min:1',
            'details.*.product_id' => 'required|exists:products,id',
            'details.*.quantity' => 'required|numeric|min:1',
            'details.*.selling_price' => 'required|numeric|min:0',
            'details.*.purchase_price' => 'required|numeric|min:0',
        ]);

        DB::beginTransaction();
        try {
            // Check stock availability
            foreach ($request->details as $detail) {
                $stock = ProductStock::where('product_id', $detail['product_id'])
                    ->where('branch_id', $request->branch_id)
                    ->first();

                if (!$stock || $stock->quantity < $detail['quantity']) {
                    $product = Product::find($detail['product_id']);
                    throw new \Exception("Stok tidak mencukupi untuk produk: {$product->name}");
                }
            }

            // Calculate amounts
            $subtotal = 0;
            foreach ($request->details as $detail) {
                $itemSubtotal = $detail['quantity'] * $detail['selling_price'];
                $itemDiscount = $detail['discount'] ?? 0;
                $subtotal += ($itemSubtotal - $itemDiscount);
            }

            $discount = $request->discount ?? 0;
            $tax = $request->tax ?? 0;
            $totalAmount = $subtotal - $discount + $tax;

            // Create sale
            $sale = Sale::create([
                'sale_number' => $this->generateSaleNumber(),
                'barcode' => $this->generateBarcode(),
                'customer_id' => $request->customer_id,
                'branch_id' => $request->branch_id,
                'cashier_id' => Auth::id(),
                'sale_date' => $request->sale_date,
                'subtotal' => $subtotal,
                'discount' => $discount,
                'tax' => $tax,
                'total_amount' => $totalAmount,
                'payment_method' => $request->payment_method,
                'status' => 'completed',
                'notes' => $request->notes,
            ]);

            // Generate QR Code and save to public folder
            $this->generateQrCode($sale);

            // Create sale details and update stock
            foreach ($request->details as $detail) {
                // Create sale detail
                $itemSubtotal = $detail['quantity'] * $detail['selling_price'];
                $itemDiscount = $detail['discount'] ?? 0;

                SaleDetail::create([
                    'sale_id' => $sale->id,
                    'product_id' => $detail['product_id'],
                    'quantity' => $detail['quantity'],
                    'selling_price' => $detail['selling_price'],
                    'purchase_price' => $detail['purchase_price'],
                    'discount' => $itemDiscount,
                    'subtotal' => $itemSubtotal - $itemDiscount,
                ]);

                // Update product stock
                $productStock = ProductStock::where('product_id', $detail['product_id'])
                    ->where('branch_id', $request->branch_id)
                    ->first();

                $productStock->decrement('quantity', $detail['quantity']);

                // Create stock movement record
                StockMovement::create([
                    'product_id' => $detail['product_id'],
                    'branch_id' => $request->branch_id,
                    'type' => 'out',
                    'quantity' => $detail['quantity'],
                    'reference_type' => 'App\Models\Sale',
                    'reference_id' => $sale->id,
                    'notes' => 'Penjualan #' . $sale->sale_number,
                    'user_id' => Auth::id(),
                ]);
            }

            DB::commit();

            return redirect()->route('sale.show', $sale->id)
                ->with(['success' => 'Data Penjualan Berhasil Disimpan! Anda dapat mencetak faktur penjualan.']);

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()
                ->with(['error' => 'Terjadi kesalahan: ' . $e->getMessage()])
                ->withInput();
        }
    }

    /**
     * Display the specified sale
     */
    public function show(Sale $sale)
    {
        $title = "sale";
        $sale->load(['customer', 'branch', 'cashier', 'saleDetails.product']);

        return view('transaction.sale.show', compact('title', 'sale'));
    }

    /**
     * Show the form for editing the specified sale
     */
    public function edit(Sale $sale)
    {
        // Only allow editing if status is pending
        // if ($sale->status !== 'pending') {
        //     return redirect()->route('sale')
        //         ->with(['error' => 'Hanya penjualan dengan status pending yang dapat diubah!']);
        // }

        $title = "sale";
        $customers = Customer::orderBy('id', 'desc')->get();
        $branches = Branch::orderBy('id', 'desc')->get();
        $products = Product::with(['brand', 'category', 'productStocks', 'productPrices'])->get();
        $sale->load(['saleDetails.product']);

        return view('transaction.sale.edit', compact('title', 'sale', 'customers', 'branches', 'products'));
    }

    /**
     * Update the specified sale
     */
    // public function update(Request $request, Sale $sale)
    // {
    //     // Only allow editing if status is pending
    //     // if ($sale->status !== 'pending') {
    //     //     return redirect()->route('sale')
    //     //         ->with(['error' => 'Hanya penjualan dengan status pending yang dapat diubah!']);
    //     // }

    //     $this->validate($request, [
    //         'branch_id' => 'required|exists:branches,id',
    //         'customer_id' => 'nullable|exists:customers,id',
    //         'sale_date' => 'required|date',
    //         'payment_method' => 'required|in:cash,transfer,credit',
    //         'details' => 'required|array|min:1',
    //         'details.*.product_id' => 'required|exists:products,id',
    //         'details.*.quantity' => 'required|numeric|min:1',
    //         'details.*.selling_price' => 'required|numeric|min:0',
    //         'details.*.purchase_price' => 'required|numeric|min:0',
    //     ]);

    //     DB::beginTransaction();
    //     try {
    //         // Calculate amounts
    //         $subtotal = 0;
    //         foreach ($request->details as $detail) {
    //             $itemSubtotal = $detail['quantity'] * $detail['selling_price'];
    //             $itemDiscount = $detail['discount'] ?? 0;
    //             $subtotal += ($itemSubtotal - $itemDiscount);
    //         }

    //         $discount = $request->discount ?? 0;
    //         $tax = $request->tax ?? 0;
    //         $totalAmount = $subtotal - $discount + $tax;

    //         // Update sale
    //         $sale->update([
    //             'customer_id' => $request->customer_id,
    //             'branch_id' => $request->branch_id,
    //             'sale_date' => $request->sale_date,
    //             'subtotal' => $subtotal,
    //             'discount' => $discount,
    //             'tax' => $tax,
    //             'total_amount' => $totalAmount,
    //             'payment_method' => $request->payment_method,
    //             'notes' => $request->notes,
    //         ]);

    //         // Delete old details
    //         $sale->saleDetails()->delete();

    //         // Create new details
    //         foreach ($request->details as $detail) {
    //             $itemSubtotal = $detail['quantity'] * $detail['selling_price'];
    //             $itemDiscount = $detail['discount'] ?? 0;

    //             SaleDetail::create([
    //                 'sale_id' => $sale->id,
    //                 'product_id' => $detail['product_id'],
    //                 'quantity' => $detail['quantity'],
    //                 'selling_price' => $detail['selling_price'],
    //                 'purchase_price' => $detail['purchase_price'],
    //                 'discount' => $itemDiscount,
    //                 'subtotal' => $itemSubtotal - $itemDiscount,
    //             ]);
    //         }

    //         DB::commit();

    //         return redirect()->route('sale')
    //             ->with(['success' => 'Data Penjualan Berhasil Diupdate!']);

    //     } catch (\Exception $e) {
    //         DB::rollBack();
    //         return redirect()->back()
    //             ->with(['error' => 'Terjadi kesalahan: ' . $e->getMessage()])
    //             ->withInput();
    //     }
    // }
public function update(Request $request, Sale $sale)
{
    $this->validate($request, [
        'branch_id' => 'required|exists:branches,id',
        'customer_id' => 'nullable|exists:customers,id',
        'sale_date' => 'required|date',
        'payment_method' => 'required|in:cash,transfer,credit',
        'details' => 'required|array|min:1',
        'details.*.product_id' => 'required|exists:products,id',
        'details.*.quantity' => 'required|numeric|min:1',
        'details.*.selling_price' => 'required|numeric|min:0',
        'details.*.purchase_price' => 'required|numeric|min:0',
    ]);

    DB::beginTransaction();

    try {

        /* ===============================
        * 1. KEMBALIKAN STOK LAMA
        * =============================== */
        foreach ($sale->saleDetails as $oldDetail) {

            $stock = ProductStock::where('product_id', $oldDetail->product_id)
                ->where('branch_id', $sale->branch_id)
                ->first();

            if ($stock) {
                $stock->increment('quantity', $oldDetail->quantity);
            }

            StockMovement::where('reference_type', Sale::class)
                ->where('reference_id', $sale->id)
                ->where('product_id', $oldDetail->product_id)
                ->delete();
        }

        /* ===============================
        * 2. HITUNG TOTAL BARU
        * =============================== */
        $subtotal = 0;
        foreach ($request->details as $detail) {
            $itemSubtotal = $detail['quantity'] * $detail['selling_price'];
            $itemDiscount = $detail['discount'] ?? 0;
            $subtotal += ($itemSubtotal - $itemDiscount);
        }

        $discount = $request->discount ?? 0;
        $tax = $request->tax ?? 0;
        $totalAmount = $subtotal - $discount + $tax;

        /* ===============================
        * 3. UPDATE SALE
        * =============================== */
        $sale->update([
            'customer_id' => $request->customer_id,
            'branch_id' => $request->branch_id,
            'sale_date' => $request->sale_date,
            'subtotal' => $subtotal,
            'discount' => $discount,
            'tax' => $tax,
            'total_amount' => $totalAmount,
            'payment_method' => $request->payment_method,
            'notes' => $request->notes,
        ]);

        /* ===============================
        * 4. HAPUS DETAIL LAMA
        * =============================== */
        $sale->saleDetails()->delete();

        /* ===============================
        * 5. SIMPAN DETAIL BARU + POTONG STOK
        * =============================== */
        foreach ($request->details as $detail) {

            $itemSubtotal = $detail['quantity'] * $detail['selling_price'];
            $itemDiscount = $detail['discount'] ?? 0;

            SaleDetail::create([
                'sale_id' => $sale->id,
                'product_id' => $detail['product_id'],
                'quantity' => $detail['quantity'],
                'selling_price' => $detail['selling_price'],
                'purchase_price' => $detail['purchase_price'],
                'discount' => $itemDiscount,
                'subtotal' => $itemSubtotal - $itemDiscount,
            ]);

            $stock = ProductStock::where('product_id', $detail['product_id'])
                ->where('branch_id', $request->branch_id)
                ->lockForUpdate()
                ->first();

            if (!$stock || $stock->quantity < $detail['quantity']) {
                throw new \Exception(
                    'Stok tidak cukup untuk produk ID: ' . $detail['product_id']
                );
            }

            $stock->decrement('quantity', $detail['quantity']);

            StockMovement::create([
                'product_id' => $detail['product_id'],
                'branch_id' => $request->branch_id,
                'type' => 'out',
                'quantity' => $detail['quantity'],
                'reference_type' => Sale::class,
                'reference_id' => $sale->id,
                'notes' => 'Update Penjualan #' . $sale->sale_number,
                'user_id' => Auth::id(),
            ]);
        }

        DB::commit();

        return redirect()->route('sale.show', $sale->id)
            ->with('success', 'Data Penjualan berhasil diperbarui');

    } catch (\Exception $e) {

        DB::rollBack();

        return redirect()->back()
            ->withInput()
            ->with('error', 'Gagal update penjualan: ' . $e->getMessage());
    }
}


    /**
     * Remove the specified sale
     */
    public function delete(Sale $sale)
    {
        // Only allow deleting if status is pending
        // if ($sale->status !== 'pending') {
        //     return redirect()->route('sale')
        //         ->with(['error' => 'Hanya penjualan dengan status pending yang dapat dihapus!']);
        // }

        DB::beginTransaction();
        try {
            // Delete sale details
            $sale->saleDetails()->delete();

            // Delete sale
            $sale->delete();

            DB::commit();

            return redirect()->route('sale')
                ->with('success', 'Data Penjualan Berhasil Dihapus');

        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->route('sale')
                ->with(['error' => 'Terjadi kesalahan: ' . $e->getMessage()]);
        }
    }

    /**
     * Get product data (price and stock) for specific branch
     */
    public function getProductData(Request $request)
    {
        $productId = $request->input('product_id');
        $branchId = $request->input('branch_id');

        if (!$productId || !$branchId) {
            return response()->json([
                'success' => false,
                'message' => 'Product ID dan Branch ID harus diisi'
            ], 400);
        }

        // Get stock
        $productStock = ProductStock::where('product_id', $productId)
            ->where('branch_id', $branchId)
            ->first();

        // Get the latest price
        $productPrice = ProductPrice::where('product_id', $productId)
            ->where('branch_id', $branchId)
            ->orderBy('effective_date', 'desc')
            ->first();

        if (!$productStock) {
            return response()->json([
                'success' => false,
                'message' => 'Stok tidak tersedia di cabang yang dipilih'
            ], 404);
        }

        if (!$productPrice) {
            return response()->json([
                'success' => false,
                'message' => 'Harga belum diatur untuk produk ini di cabang yang dipilih'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'stock' => $productStock->quantity,
            'purchase_price' => $productPrice->purchase_price,
            'selling_price' => $productPrice->selling_price,
        ]);
    }

    /**
     * Get products by branch (only products that have stock in the selected branch)
     */
    public function getProductsByBranch(Request $request)
    {
        $branchId = $request->input('branch_id');

        if (!$branchId) {
            // If no branch selected, return all products
            $products = Product::with(['brand', 'category'])->get();

            return response()->json([
                'success' => true,
                'products' => $products->map(function($product) {
                    return [
                        'id' => $product->id,
                        'name' => $product->name,
                        'unit' => $product->unit,
                        'brand_name' => $product->brand->name ?? '-',
                        'stock' => 0,
                    ];
                })
            ]);
        } else {
            // Get only products that have stock in the selected branch
            $productStocks = ProductStock::where('branch_id', $branchId)
                ->with(['product.brand', 'product.category'])
                ->get();

            return response()->json([
                'success' => true,
                'products' => $productStocks->map(function($productStock) {
                    return [
                        'id' => $productStock->product->id,
                        'name' => $productStock->product->name,
                        'unit' => $productStock->product->unit,
                        'brand_name' => $productStock->product->brand->name ?? '-',
                        'stock' => $productStock->quantity ?? 0,
                    ];
                })
            ]);
        }
    }

    /**
     * Find product by barcode for QR code scanning
     */
    public function findProductByBarcode(Request $request)
    {
        $barcode = $request->input('barcode');
        $branchId = $request->input('branch_id');

        if (!$barcode) {
            return response()->json([
                'success' => false,
                'message' => 'Barcode harus diisi'
            ], 400);
        }

        if (!$branchId) {
            return response()->json([
                'success' => false,
                'message' => 'Silakan pilih cabang terlebih dahulu'
            ], 400);
        }

        // Find product by barcode
        $product = Product::with(['brand', 'category'])
            ->where('barcode', $barcode)
            ->first();

        if (!$product) {
            return response()->json([
                'success' => false,
                'message' => 'Produk dengan barcode "' . $barcode . '" tidak ditemukan!'
            ], 404);
        }

        // Check if product has stock in selected branch
        $productStock = ProductStock::where('product_id', $product->id)
            ->where('branch_id', $branchId)
            ->first();

        if (!$productStock || $productStock->quantity <= 0) {
            return response()->json([
                'success' => false,
                'message' => 'Stok produk "' . $product->name . '" tidak tersedia di cabang ini!'
            ], 404);
        }

        // Get the latest price
        $productPrice = ProductPrice::where('product_id', $product->id)
            ->where('branch_id', $branchId)
            ->orderBy('effective_date', 'desc')
            ->first();

        if (!$productPrice) {
            return response()->json([
                'success' => false,
                'message' => 'Harga belum diatur untuk produk ini di cabang yang dipilih'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'product' => [
                'id' => $product->id,
                'name' => $product->name,
                'unit' => $product->unit,
                'brand_name' => $product->brand->name ?? '-',
                'stock' => $productStock->quantity,
                'purchase_price' => $productPrice->purchase_price,
                'selling_price' => $productPrice->selling_price,
            ]
        ]);
    }

    /**
     * Generate unique sale number
     */
    private function generateSaleNumber()
    {
        $prefix = 'SO';
        $date = date('Ymd');
        $lastSale = Sale::whereDate('created_at', today())
            ->orderBy('id', 'desc')
            ->first();

        if ($lastSale) {
            $lastNumber = intval(substr($lastSale->sale_number, -4));
            $newNumber = $lastNumber + 1;
        } else {
            $newNumber = 1;
        }

        return $prefix . $date . str_pad($newNumber, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Generate unique barcode for sale
     */
    private function generateBarcode()
    {
        $length = 100; // total panjang karakter random (tanpa SAL + date)
        $characters = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!@#$%^&*()_-+=[]{}|;:,.<>?';
        $charactersLength = strlen($characters);

        do {
            $randomString = '';
            for ($i = 0; $i < $length; $i++) {
                $randomString .= $characters[random_int(0, $charactersLength - 1)];
            }

            // contoh hasil: SAL250103 + 100 random characters
            $barcode = 'SAL' . date('ymd') . $randomString;

        } while (Sale::where('barcode', $barcode)->exists());

        return $barcode;
    }

    /**
     * Generate QR Code for sale
     */
    private function generateQrCode(Sale $sale)
    {
        try {
            // Create QR code folder if not exists
            $qrPath = public_path('qrcodes/sales');
            if (!file_exists($qrPath)) {
                mkdir($qrPath, 0755, true);
            }

            // Generate QR Code using BaconQrCode v3 with SVG (for display)
            $svgRenderer = new ImageRenderer(
                new RendererStyle(300, 10),
                new SvgImageBackEnd()
            );

            $svgWriter = new Writer($svgRenderer);

            // Save QR Code as SVG
            $svgFileName = 'sale_' . $sale->id . '.svg';
            $svgFilePath = $qrPath . '/' . $svgFileName;

            $svgWriter->writeFile($sale->barcode, $svgFilePath);

            // Generate PNG version for download using QR code matrix
            $pngFileName = 'sale_' . $sale->id . '.png';
            $pngFilePath = $qrPath . '/' . $pngFileName;

            // Encode data to get QR matrix
            $qrCode = Encoder::encode($sale->barcode, ErrorCorrectionLevel::L(), 'UTF-8');
            $matrix = $qrCode->getMatrix();

            $matrixWidth = $matrix->getWidth();
            $matrixHeight = $matrix->getHeight();

            // Set scale for better quality (each module = 10 pixels)
            $scale = 10;
            $margin = 20; // margin in pixels

            $imageWidth = $matrixWidth * $scale + (2 * $margin);
            $imageHeight = $matrixHeight * $scale + (2 * $margin);

            // Create PNG image
            $img = imagecreatetruecolor($imageWidth, $imageHeight);
            $white = imagecolorallocate($img, 255, 255, 255);
            $black = imagecolorallocate($img, 0, 0, 0);
            imagefill($img, 0, 0, $white);

            // Draw QR code modules
            for ($y = 0; $y < $matrixHeight; $y++) {
                for ($x = 0; $x < $matrixWidth; $x++) {
                    if ($matrix->get($x, $y) == 1) {
                        imagefilledrectangle(
                            $img,
                            $margin + ($x * $scale),
                            $margin + ($y * $scale),
                            $margin + (($x + 1) * $scale) - 1,
                            $margin + (($y + 1) * $scale) - 1,
                            $black
                        );
                    }
                }
            }

            // Save as PNG
            imagepng($img, $pngFilePath, 0); // 0 = no compression for best quality
            imagedestroy($img);

            return true;
        } catch (\Exception $e) {
            \Log::error('Failed to generate QR Code for sale ' . $sale->id . ': ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Find sale by barcode (for QR code scanning)
     */
    public function findSaleByBarcode(Request $request)
    {
        $barcode = $request->input('barcode');

        if (!$barcode) {
            return response()->json([
                'success' => false,
                'message' => 'Barcode tidak boleh kosong'
            ], 400);
        }

        $sale = Sale::with(['customer', 'branch', 'cashier'])
            ->where('barcode', $barcode)
            ->first();

        if (!$sale) {
            return response()->json([
                'success' => false,
                'message' => 'Penjualan tidak ditemukan'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'sale' => [
                'id' => $sale->id,
                'sale_number' => $sale->sale_number,
                'barcode' => $sale->barcode,
                'customer_name' => $sale->customer ? $sale->customer->name : 'Umum',
                'branch_name' => $sale->branch->name,
                'cashier_name' => $sale->cashier->name,
                'sale_date' => $sale->sale_date->format('d/m/Y'),
                'total_amount' => number_format($sale->total_amount, 0, ',', '.'),
                'payment_method' => $sale->payment_method,
                'status' => $sale->status,
                'url' => route('sale.show', $sale->id)
            ]
        ]);
    }

    /**
     * Download QR Code as JPG
     */
    public function downloadQrCodeJpg(Sale $sale)
    {
        try {
            // Generate QR Code matrix using BaconQrCode
            $qrCode = \BaconQrCode\Encoder\Encoder::encode(
                $sale->barcode,
                \BaconQrCode\Common\ErrorCorrectionLevel::L(),
                'UTF-8'
            );
            $matrix = $qrCode->getMatrix();

            $moduleSize = 15;
            $quietZone = 4;
            $matrixWidth = $matrix->getWidth();

            $imageWidth = ($matrixWidth + ($quietZone * 2)) * $moduleSize;

            // Create image with white background
            $img = imagecreatetruecolor($imageWidth, $imageWidth);
            $white = imagecolorallocate($img, 255, 255, 255);
            $black = imagecolorallocate($img, 0, 0, 0);

            // Fill background
            imagefill($img, 0, 0, $white);

            // Draw QR code modules
            for ($y = 0; $y < $matrixWidth; $y++) {
                for ($x = 0; $x < $matrixWidth; $x++) {
                    if ($matrix->get($x, $y) === 1) {
                        $xPos = ($x + $quietZone) * $moduleSize;
                        $yPos = ($y + $quietZone) * $moduleSize;
                        imagefilledrectangle($img, $xPos, $yPos, $xPos + $moduleSize - 1, $yPos + $moduleSize - 1, $black);
                    }
                }
            }

            // Save to temporary file
            $tempJpgPath = sys_get_temp_dir() . '/qr_sale_' . $sale->id . '_' . time() . '.jpg';
            imagejpeg($img, $tempJpgPath, 95);
            imagedestroy($img);

            // Download file and delete after
            $fileName = 'QRCode-Sale-' . $sale->sale_number . '.jpg';

            return response()->download($tempJpgPath, $fileName, [
                'Content-Type' => 'image/jpeg'
            ])->deleteFileAfterSend(true);

        } catch (\Exception $e) {
            abort(500, 'Gagal generate QR Code JPG: ' . $e->getMessage());
        }
    }

    /**
     * Export sale invoice to PDF
     */
    public function exportInvoicePdf(Sale $sale)
    {
        $sale->load(['customer', 'branch', 'cashier', 'saleDetails.product']);

        $pdf = Pdf::loadView('transaction.sale.invoice-pdf', compact('sale'));
        $pdf->setPaper('a4', 'portrait');

        return $pdf->stream('Faktur-Penjualan-' . $sale->sale_number . '.pdf');
    }
}
