<?php

namespace App\Http\Controllers\Report;

use App\Http\Controllers\Controller;
use App\Models\Service;
use App\Models\ServiceDetail;
use App\Models\User;
use App\Models\Branch;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Barryvdh\DomPDF\Facade\Pdf;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

class ServiceReportController extends Controller
{
    public function __construct()
    {
        $this->middleware(['permission:service-reports.index']);
    }

    /**
     * Laporan Service Berdasarkan Status
     */
    public function statusReport(Request $request)
    {
        $title = "service_status";
        $branches = Branch::orderBy('name', 'asc')->get();

        // Default filter: bulan ini
        $startDate = $request->start_date ?? Carbon::now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->end_date ?? Carbon::now()->endOfMonth()->format('Y-m-d');
        $branchId = $request->branch_id;

        // Query service
        $query = Service::with(['customer', 'branch', 'technician'])
            ->whereBetween('service_date', [$startDate, $endDate]);

        if ($branchId) {
            $query->where('branch_id', $branchId);
        }

        $services = $query->orderBy('service_date', 'desc')->get();

        // Statistik per status
        $statusStats = $services->groupBy('status')->map(function ($items) {
            return [
                'count' => $items->count(),
                'total_value' => $items->sum('total_cost')
            ];
        });

        // Service tertunda (>3 hari dari tanggal service, belum selesai)
        $delayedServices = $services->filter(function ($service) {
            return !in_array($service->status, ['selesai', 'dibatalkan'])
                && Carbon::parse($service->service_date)->addDays(3)->isPast();
        });

        // Service siap diambil tapi belum diambil
        $readyToPickup = $services->filter(function ($service) {
            return $service->status === 'siap diambil';
        });

        return view('report.service.status', compact(
            'title', 'branches', 'services', 'statusStats',
            'delayedServices', 'readyToPickup',
            'startDate', 'endDate', 'branchId'
        ));
    }

    /**
     * Laporan Kinerja Teknisi
     */
    public function technicianReport(Request $request)
    {
        $title = "service_technician";
        $technicians = User::orderBy('name', 'asc')->get();
        $branches = Branch::orderBy('name', 'asc')->get();

        // Default filter: bulan ini
        $startDate = $request->start_date ?? Carbon::now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->end_date ?? Carbon::now()->endOfMonth()->format('Y-m-d');
        $technicianId = $request->technician_id;
        $branchId = $request->branch_id;

        // Query service
        $query = Service::with(['customer', 'branch', 'technician'])
            ->whereBetween('service_date', [$startDate, $endDate]);

        if ($technicianId) {
            $query->where('technician_id', $technicianId);
        }

        if ($branchId) {
            $query->where('branch_id', $branchId);
        }

        $services = $query->get();

        // Statistik per teknisi
        $technicianStats = $services->groupBy('technician_id')->map(function ($items, $techId) {
            $technician = User::find($techId);
            $completed = $items->where('status', 'selesai')->count();
            $cancelled = $items->where('status', 'dibatalkan')->count();
            $total = $items->count();

            // Hitung rata-rata durasi (dari service_date ke completed_at)
            $completedServices = $items->filter(function ($service) {
                return $service->completed_at !== null;
            });

            $avgDuration = 0;
            if ($completedServices->count() > 0) {
                $totalDays = $completedServices->sum(function ($service) {
                    return Carbon::parse($service->service_date)->diffInDays($service->completed_at);
                });
                $avgDuration = round($totalDays / $completedServices->count(), 1);
            }

            return [
                'technician_name' => $technician ? $technician->name : 'Tidak ada teknisi',
                'total_services' => $total,
                'completed' => $completed,
                'cancelled' => $cancelled,
                'in_progress' => $total - $completed - $cancelled,
                'success_rate' => $total > 0 ? round(($completed / $total) * 100, 1) : 0,
                'avg_duration_days' => $avgDuration,
                'total_revenue' => $items->where('status', 'selesai')->sum('total_cost')
            ];
        })->sortByDesc('total_services');

        return view('report.service.technician', compact(
            'title', 'technicians', 'branches', 'services', 'technicianStats',
            'startDate', 'endDate', 'technicianId', 'branchId'
        ));
    }

    /**
     * Laporan Pendapatan Service
     */
    public function revenueReport(Request $request)
    {
        $title = "service_revenue";
        $branches = Branch::orderBy('name', 'asc')->get();

        // Default filter: bulan ini
        $startDate = $request->start_date ?? Carbon::now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->end_date ?? Carbon::now()->endOfMonth()->format('Y-m-d');
        $branchId = $request->branch_id;

        // Query service
        $query = Service::with(['customer', 'branch', 'technician'])
            ->whereBetween('service_date', [$startDate, $endDate]);

        if ($branchId) {
            $query->where('branch_id', $branchId);
        }

        $services = $query->orderBy('service_date', 'desc')->get();

        // Total pendapatan
        $totalServiceFee = $services->sum('service_fee');
        $totalPartsCost = $services->sum('parts_cost');
        $totalRevenue = $services->sum('total_cost');
        $totalDownPayment = $services->sum('down_payment');
        $totalRemaining = $services->sum('remaining_payment');

        // Pendapatan per cabang
        $branchRevenue = $services->groupBy('branch_id')->map(function ($items, $branchId) {
            $branch = Branch::find($branchId);
            return [
                'branch_name' => $branch ? $branch->name : 'Unknown',
                'total_services' => $items->count(),
                'total_revenue' => $items->sum('total_cost'),
                'total_service_fee' => $items->sum('service_fee'),
                'total_parts_cost' => $items->sum('parts_cost')
            ];
        })->sortByDesc('total_revenue');

        return view('report.service.revenue', compact(
            'title', 'branches', 'services',
            'totalServiceFee', 'totalPartsCost', 'totalRevenue', 'totalDownPayment', 'totalRemaining',
            'branchRevenue',
            'startDate', 'endDate', 'branchId'
        ));
    }

    /**
     * Export Laporan Status to PDF
     */
    public function exportStatusPdf(Request $request)
    {
        $startDate = $request->start_date ?? Carbon::now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->end_date ?? Carbon::now()->endOfMonth()->format('Y-m-d');
        $branchId = $request->branch_id;

        $query = Service::with(['customer', 'branch', 'technician'])
            ->whereBetween('service_date', [$startDate, $endDate]);

        if ($branchId) {
            $query->where('branch_id', $branchId);
            $branchName = Branch::find($branchId)->name;
        } else {
            $branchName = 'Semua Cabang';
        }

        $services = $query->orderBy('service_date', 'desc')->get();

        $statusStats = $services->groupBy('status')->map(function ($items) {
            return [
                'count' => $items->count(),
                'total_value' => $items->sum('total_cost')
            ];
        });

        $pdf = Pdf::loadView('report.service.pdf.status', compact(
            'services', 'statusStats', 'startDate', 'endDate', 'branchName'
        ));

        return $pdf->download('laporan-status-service-' . $startDate . '-' . $endDate . '.pdf');
    }

    /**
     * Export Laporan Status to Excel
     */
    public function exportStatusExcel(Request $request)
    {
        $startDate = $request->start_date ?? Carbon::now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->end_date ?? Carbon::now()->endOfMonth()->format('Y-m-d');
        $branchId = $request->branch_id;

        $query = Service::with(['customer', 'branch', 'technician'])
            ->whereBetween('service_date', [$startDate, $endDate]);

        if ($branchId) {
            $query->where('branch_id', $branchId);
            $branchName = Branch::find($branchId)->name;
        } else {
            $branchName = 'Semua Cabang';
        }

        $services = $query->orderBy('service_date', 'desc')->get();

        $statusStats = $services->groupBy('status')->map(function ($items) {
            return [
                'count' => $items->count(),
                'total_value' => $items->sum('total_cost')
            ];
        });

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header
        $sheet->setCellValue('A1', 'LAPORAN SERVICE BERDASARKAN STATUS');
        $sheet->mergeCells('A1:H1');
        $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
        $sheet->getStyle('A1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        $sheet->setCellValue('A2', 'Periode: ' . Carbon::parse($startDate)->format('d/m/Y') . ' - ' . Carbon::parse($endDate)->format('d/m/Y'));
        $sheet->mergeCells('A2:H2');
        $sheet->getStyle('A2')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        $sheet->setCellValue('A3', 'Cabang: ' . $branchName);
        $sheet->mergeCells('A3:H3');
        $sheet->getStyle('A3')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        // Ringkasan Section
        $row = 5;
        $sheet->setCellValue('A' . $row, 'RINGKASAN');
        $sheet->mergeCells('A' . $row . ':B' . $row);
        $sheet->getStyle('A' . $row)->getFont()->setBold(true)->setSize(12);
        $sheet->getStyle('A' . $row)->getFill()
            ->setFillType(Fill::FILL_SOLID)
            ->getStartColor()->setARGB('FFE0E0E0');

        $row++;
        $sheet->setCellValue('A' . $row, 'Total Service:');
        $sheet->setCellValue('B' . $row, $services->count());

        $row++;
        $sheet->setCellValue('A' . $row, 'Selesai:');
        $sheet->setCellValue('B' . $row, $services->where('status', 'selesai')->count());

        $row++;
        $tertunda = $services->filter(function($s) {
            return Carbon::parse($s->service_date)->diffInDays(Carbon::now()) > 3
                && !in_array($s->status, ['selesai', 'dibatalkan']);
        })->count();
        $sheet->setCellValue('A' . $row, 'Tertunda (> 3 hari):');
        $sheet->setCellValue('B' . $row, $tertunda);

        $row++;
        $sheet->setCellValue('A' . $row, 'Siap Diambil:');
        $sheet->setCellValue('B' . $row, $services->where('status', 'siap_diambil')->count());

        // Statistik Per Status Section
        $row += 2;
        $sheet->setCellValue('A' . $row, 'STATISTIK PER STATUS');
        $sheet->mergeCells('A' . $row . ':C' . $row);
        $sheet->getStyle('A' . $row)->getFont()->setBold(true)->setSize(12);

        $row++;
        $statusHeaders = ['Status', 'Jumlah', 'Total Nilai'];
        $col = 'A';
        foreach ($statusHeaders as $header) {
            $sheet->setCellValue($col . $row, $header);
            $sheet->getStyle($col . $row)->getFont()->setBold(true);
            $sheet->getStyle($col . $row)->getFill()
                ->setFillType(Fill::FILL_SOLID)
                ->getStartColor()->setARGB('FF4CAF50');
            $sheet->getStyle($col . $row)->getFont()->getColor()->setARGB('FFFFFFFF');
            $col++;
        }

        $row++;
        $startStatusRow = $row;
        foreach ($statusStats as $status => $stat) {
            $sheet->setCellValue('A' . $row, ucwords($status));
            $sheet->setCellValue('B' . $row, $stat['count']);
            $sheet->setCellValue('C' . $row, 'Rp ' . number_format($stat['total_value'], 0, ',', '.'));
            $row++;
        }

        // Total Row for Statistik Per Status
        $sheet->setCellValue('A' . $row, 'TOTAL');
        $sheet->setCellValue('B' . $row, $services->count());
        $sheet->setCellValue('C' . $row, 'Rp ' . number_format($services->sum('total_cost'), 0, ',', '.'));
        $sheet->getStyle('A' . $row . ':C' . $row)->getFont()->setBold(true);
        $sheet->getStyle('A' . $row . ':C' . $row)->getFill()
            ->setFillType(Fill::FILL_SOLID)
            ->getStartColor()->setARGB('FFEEEEEE');

        // Borders for Statistik Per Status table
        $sheet->getStyle('A' . ($startStatusRow - 1) . ':C' . $row)->getBorders()->getAllBorders()
            ->setBorderStyle(Border::BORDER_THIN);

        // Detail Service Section
        $row += 2;
        $sheet->setCellValue('A' . $row, 'DETAIL SERVICE');
        $sheet->mergeCells('A' . $row . ':H' . $row);
        $sheet->getStyle('A' . $row)->getFont()->setBold(true)->setSize(12);

        // Table Header
        $row++;
        $headers = ['No. Service', 'Tanggal', 'Customer', 'Device', 'Status', 'Cabang', 'Teknisi', 'Total'];
        $col = 'A';
        foreach ($headers as $header) {
            $sheet->setCellValue($col . $row, $header);
            $sheet->getStyle($col . $row)->getFont()->setBold(true);
            $sheet->getStyle($col . $row)->getFill()
                ->setFillType(Fill::FILL_SOLID)
                ->getStartColor()->setARGB('FF4CAF50');
            $sheet->getStyle($col . $row)->getFont()->getColor()->setARGB('FFFFFFFF');
            $col++;
        }

        // Data
        $startDetailRow = $row + 1;
        $row++;
        foreach ($services as $service) {
            $sheet->setCellValue('A' . $row, $service->service_number);
            $sheet->setCellValue('B' . $row, $service->service_date->format('d/m/Y'));
            $sheet->setCellValue('C' . $row, $service->customer->name);
            $sheet->setCellValue('D' . $row, $service->device_brand . ' ' . $service->device_model);
            $sheet->setCellValue('E' . $row, ucwords($service->status));
            $sheet->setCellValue('F' . $row, $service->branch->name);
            $sheet->setCellValue('G' . $row, $service->technician->name ?? '-');
            $sheet->setCellValue('H' . $row, 'Rp ' . number_format($service->total_cost, 0, ',', '.'));
            $row++;
        }

        // Borders for Detail Service table
        $sheet->getStyle('A' . ($startDetailRow - 1) . ':H' . ($row - 1))->getBorders()->getAllBorders()
            ->setBorderStyle(Border::BORDER_THIN);

        // Auto size columns
        foreach (range('A', 'H') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'laporan-status-service-' . $startDate . '-' . $endDate . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }

    /**
     * Export Laporan Teknisi to PDF
     */
    public function exportTechnicianPdf(Request $request)
    {
        $startDate = $request->start_date ?? Carbon::now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->end_date ?? Carbon::now()->endOfMonth()->format('Y-m-d');
        $technicianId = $request->technician_id;
        $branchId = $request->branch_id;

        $query = Service::with(['customer', 'branch', 'technician'])
            ->whereBetween('service_date', [$startDate, $endDate]);

        if ($technicianId) {
            $query->where('technician_id', $technicianId);
            $technicianName = User::find($technicianId)->name;
        } else {
            $technicianName = 'Semua Teknisi';
        }

        if ($branchId) {
            $query->where('branch_id', $branchId);
            $branchName = Branch::find($branchId)->name;
        } else {
            $branchName = 'Semua Cabang';
        }

        $services = $query->get();

        $technicianStats = $services->groupBy('technician_id')->map(function ($items, $techId) {
            $technician = User::find($techId);
            $completed = $items->where('status', 'selesai')->count();
            $cancelled = $items->where('status', 'dibatalkan')->count();
            $total = $items->count();

            $completedServices = $items->filter(fn($s) => $s->completed_at !== null);
            $avgDuration = 0;
            if ($completedServices->count() > 0) {
                $totalDays = $completedServices->sum(fn($s) => Carbon::parse($s->service_date)->diffInDays($s->completed_at));
                $avgDuration = round($totalDays / $completedServices->count(), 1);
            }

            return [
                'technician_name' => $technician ? $technician->name : 'Tidak ada teknisi',
                'total_services' => $total,
                'completed' => $completed,
                'cancelled' => $cancelled,
                'in_progress' => $total - $completed - $cancelled,
                'success_rate' => $total > 0 ? round(($completed / $total) * 100, 1) : 0,
                'avg_duration_days' => $avgDuration,
                'total_revenue' => $items->where('status', 'selesai')->sum('total_cost')
            ];
        })->sortByDesc('total_services');

        $pdf = Pdf::loadView('report.service.pdf.technician', compact(
            'services', 'technicianStats', 'startDate', 'endDate', 'technicianName', 'branchName'
        ));

        return $pdf->download('laporan-teknisi-service-' . $startDate . '-' . $endDate . '.pdf');
    }

    /**
     * Export Laporan Teknisi to Excel
     */
    public function exportTechnicianExcel(Request $request)
    {
        $startDate = $request->start_date ?? Carbon::now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->end_date ?? Carbon::now()->endOfMonth()->format('Y-m-d');
        $technicianId = $request->technician_id;
        $branchId = $request->branch_id;

        $query = Service::with(['customer', 'branch', 'technician'])
            ->whereBetween('service_date', [$startDate, $endDate]);

        if ($technicianId) {
            $query->where('technician_id', $technicianId);
            $technicianName = User::find($technicianId)->name;
        } else {
            $technicianName = 'Semua Teknisi';
        }

        if ($branchId) {
            $query->where('branch_id', $branchId);
            $branchName = Branch::find($branchId)->name;
        } else {
            $branchName = 'Semua Cabang';
        }

        $services = $query->get();

        $technicianStats = $services->groupBy('technician_id')->map(function ($items, $techId) {
            $technician = User::find($techId);
            $completed = $items->where('status', 'selesai')->count();
            $cancelled = $items->where('status', 'dibatalkan')->count();
            $total = $items->count();

            $completedServices = $items->filter(fn($s) => $s->completed_at !== null);
            $avgDuration = 0;
            if ($completedServices->count() > 0) {
                $totalDays = $completedServices->sum(fn($s) => Carbon::parse($s->service_date)->diffInDays($s->completed_at));
                $avgDuration = round($totalDays / $completedServices->count(), 1);
            }

            return [
                'technician_name' => $technician ? $technician->name : 'Tidak ada teknisi',
                'total_services' => $total,
                'completed' => $completed,
                'cancelled' => $cancelled,
                'in_progress' => $total - $completed - $cancelled,
                'success_rate' => $total > 0 ? round(($completed / $total) * 100, 1) : 0,
                'avg_duration_days' => $avgDuration,
                'total_revenue' => $items->where('status', 'selesai')->sum('total_cost')
            ];
        })->sortByDesc('total_services');

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header
        $sheet->setCellValue('A1', 'LAPORAN KINERJA TEKNISI');
        $sheet->mergeCells('A1:H1');
        $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
        $sheet->getStyle('A1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        $sheet->setCellValue('A2', 'Periode: ' . Carbon::parse($startDate)->format('d/m/Y') . ' - ' . Carbon::parse($endDate)->format('d/m/Y'));
        $sheet->mergeCells('A2:H2');
        $sheet->getStyle('A2')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        $sheet->setCellValue('A3', 'Teknisi: ' . $technicianName . ' | Cabang: ' . $branchName);
        $sheet->mergeCells('A3:H3');
        $sheet->getStyle('A3')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        // Summary Section
        $row = 5;
        $sheet->setCellValue('A' . $row, 'RINGKASAN');
        $sheet->mergeCells('A' . $row . ':B' . $row);
        $sheet->getStyle('A' . $row)->getFont()->setBold(true)->setSize(12);
        $sheet->getStyle('A' . $row)->getFill()
            ->setFillType(Fill::FILL_SOLID)
            ->getStartColor()->setARGB('FFE0E0E0');

        $row++;
        $sheet->setCellValue('A' . $row, 'Total Service:');
        $sheet->setCellValue('B' . $row, $services->count());

        $row++;
        $sheet->setCellValue('A' . $row, 'Selesai:');
        $sheet->setCellValue('B' . $row, $services->where('status', 'selesai')->count());

        $row++;
        $sheet->setCellValue('A' . $row, 'Jumlah Teknisi:');
        $sheet->setCellValue('B' . $row, $technicianStats->count());

        $row++;
        $sheet->setCellValue('A' . $row, 'Total Pendapatan:');
        $sheet->setCellValue('B' . $row, 'Rp ' . number_format($services->where('status', 'selesai')->sum('total_cost'), 0, ',', '.'));

        // Statistik Per Teknisi Section
        $row += 2;
        $sheet->setCellValue('A' . $row, 'STATISTIK PER TEKNISI');
        $sheet->mergeCells('A' . $row . ':H' . $row);
        $sheet->getStyle('A' . $row)->getFont()->setBold(true)->setSize(12);

        // Table Header
        $row++;
        $headers = ['Teknisi', 'Total Service', 'Selesai', 'Dalam Proses', 'Dibatalkan', 'Success Rate', 'Avg Durasi', 'Total Pendapatan'];
        $col = 'A';
        foreach ($headers as $header) {
            $sheet->setCellValue($col . $row, $header);
            $sheet->getStyle($col . $row)->getFont()->setBold(true);
            $sheet->getStyle($col . $row)->getFill()
                ->setFillType(Fill::FILL_SOLID)
                ->getStartColor()->setARGB('FF2196F3');
            $sheet->getStyle($col . $row)->getFont()->getColor()->setARGB('FFFFFFFF');
            $col++;
        }

        // Data
        $startDataRow = $row + 1;
        $row++;
        foreach ($technicianStats as $stat) {
            $sheet->setCellValue('A' . $row, $stat['technician_name']);
            $sheet->setCellValue('B' . $row, $stat['total_services']);
            $sheet->setCellValue('C' . $row, $stat['completed']);
            $sheet->setCellValue('D' . $row, $stat['in_progress']);
            $sheet->setCellValue('E' . $row, $stat['cancelled']);
            $sheet->setCellValue('F' . $row, $stat['success_rate'] . '%');
            $sheet->setCellValue('G' . $row, $stat['avg_duration_days'] . ' hari');
            $sheet->setCellValue('H' . $row, 'Rp ' . number_format($stat['total_revenue'], 0, ',', '.'));
            $row++;
        }

        // Total Row for Statistik Per Teknisi
        $sheet->setCellValue('A' . $row, 'TOTAL');
        $sheet->setCellValue('B' . $row, $technicianStats->sum('total_services'));
        $sheet->setCellValue('C' . $row, $technicianStats->sum('completed'));
        $sheet->setCellValue('D' . $row, $technicianStats->sum('in_progress'));
        $sheet->setCellValue('E' . $row, $technicianStats->sum('cancelled'));
        $sheet->setCellValue('F' . $row, '-');
        $sheet->setCellValue('G' . $row, '-');
        $sheet->setCellValue('H' . $row, 'Rp ' . number_format($technicianStats->sum('total_revenue'), 0, ',', '.'));
        $sheet->getStyle('A' . $row . ':H' . $row)->getFont()->setBold(true);
        $sheet->getStyle('A' . $row . ':H' . $row)->getFill()
            ->setFillType(Fill::FILL_SOLID)
            ->getStartColor()->setARGB('FFEEEEEE');

        // Borders for Statistik Per Teknisi table
        $sheet->getStyle('A' . ($startDataRow - 1) . ':H' . $row)->getBorders()->getAllBorders()
            ->setBorderStyle(Border::BORDER_THIN);

        // Detail Service Section
        $row += 2;
        $sheet->setCellValue('A' . $row, 'DETAIL SERVICE');
        $sheet->mergeCells('A' . $row . ':G' . $row);
        $sheet->getStyle('A' . $row)->getFont()->setBold(true)->setSize(12);

        // Detail Service Table Header
        $row++;
        $detailHeaders = ['No. Service', 'Tanggal', 'Customer', 'Device', 'Teknisi', 'Status', 'Total'];
        $col = 'A';
        foreach ($detailHeaders as $header) {
            $sheet->setCellValue($col . $row, $header);
            $sheet->getStyle($col . $row)->getFont()->setBold(true);
            $sheet->getStyle($col . $row)->getFill()
                ->setFillType(Fill::FILL_SOLID)
                ->getStartColor()->setARGB('FF2196F3');
            $sheet->getStyle($col . $row)->getFont()->getColor()->setARGB('FFFFFFFF');
            $col++;
        }

        // Detail Service Data
        $startDetailRow = $row + 1;
        $row++;
        foreach ($services as $service) {
            $sheet->setCellValue('A' . $row, $service->service_number);
            $sheet->setCellValue('B' . $row, $service->service_date->format('d/m/Y'));
            $sheet->setCellValue('C' . $row, $service->customer->name);
            $sheet->setCellValue('D' . $row, $service->device_brand . ' ' . $service->device_model);
            $sheet->setCellValue('E' . $row, $service->technician->name ?? '-');
            $sheet->setCellValue('F' . $row, ucwords($service->status));
            $sheet->setCellValue('G' . $row, 'Rp ' . number_format($service->total_cost, 0, ',', '.'));
            $row++;
        }

        // Borders for Detail Service table
        $sheet->getStyle('A' . ($startDetailRow - 1) . ':G' . ($row - 1))->getBorders()->getAllBorders()
            ->setBorderStyle(Border::BORDER_THIN);

        // Auto size columns
        foreach (range('A', 'H') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'laporan-teknisi-service-' . $startDate . '-' . $endDate . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }

    /**
     * Export Laporan Pendapatan to PDF
     */
    public function exportRevenuePdf(Request $request)
    {
        $startDate = $request->start_date ?? Carbon::now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->end_date ?? Carbon::now()->endOfMonth()->format('Y-m-d');
        $branchId = $request->branch_id;

        $query = Service::with(['customer', 'branch', 'technician'])
            ->whereBetween('service_date', [$startDate, $endDate]);

        if ($branchId) {
            $query->where('branch_id', $branchId);
            $branchName = Branch::find($branchId)->name;
        } else {
            $branchName = 'Semua Cabang';
        }

        $services = $query->orderBy('service_date', 'desc')->get();

        $totalServiceFee = $services->sum('service_fee');
        $totalPartsCost = $services->sum('parts_cost');
        $totalRevenue = $services->sum('total_cost');
        $totalDownPayment = $services->sum('down_payment');
        $totalRemaining = $services->sum('remaining_payment');

        $branchRevenue = $services->groupBy('branch_id')->map(function ($items, $branchId) {
            $branch = Branch::find($branchId);
            return [
                'branch_name' => $branch ? $branch->name : 'Unknown',
                'total_services' => $items->count(),
                'total_revenue' => $items->sum('total_cost'),
                'total_service_fee' => $items->sum('service_fee'),
                'total_parts_cost' => $items->sum('parts_cost')
            ];
        })->sortByDesc('total_revenue');

        $pdf = Pdf::loadView('report.service.pdf.revenue', compact(
            'services', 'totalServiceFee', 'totalPartsCost', 'totalRevenue',
            'totalDownPayment', 'totalRemaining', 'branchRevenue',
            'startDate', 'endDate', 'branchName'
        ));

        return $pdf->download('laporan-pendapatan-service-' . $startDate . '-' . $endDate . '.pdf');
    }

    /**
     * Export Laporan Pendapatan to Excel
     */
    public function exportRevenueExcel(Request $request)
    {
        $startDate = $request->start_date ?? Carbon::now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->end_date ?? Carbon::now()->endOfMonth()->format('Y-m-d');
        $branchId = $request->branch_id;

        $query = Service::with(['customer', 'branch', 'technician'])
            ->whereBetween('service_date', [$startDate, $endDate]);

        if ($branchId) {
            $query->where('branch_id', $branchId);
            $branchName = Branch::find($branchId)->name;
        } else {
            $branchName = 'Semua Cabang';
        }

        $services = $query->orderBy('service_date', 'desc')->get();

        $totalServiceFee = $services->sum('service_fee');
        $totalPartsCost = $services->sum('parts_cost');
        $totalRevenue = $services->sum('total_cost');
        $totalDownPayment = $services->sum('down_payment');
        $totalRemaining = $services->sum('remaining_payment');

        $branchRevenue = $services->groupBy('branch_id')->map(function ($items, $branchId) {
            $branch = Branch::find($branchId);
            return [
                'branch_name' => $branch ? $branch->name : 'Unknown',
                'total_services' => $items->count(),
                'total_revenue' => $items->sum('total_cost'),
                'total_service_fee' => $items->sum('service_fee'),
                'total_parts_cost' => $items->sum('parts_cost')
            ];
        })->sortByDesc('total_revenue');

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Header
        $sheet->setCellValue('A1', 'LAPORAN PENDAPATAN SERVICE');
        $sheet->mergeCells('A1:J1');
        $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
        $sheet->getStyle('A1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        $sheet->setCellValue('A2', 'Periode: ' . Carbon::parse($startDate)->format('d/m/Y') . ' - ' . Carbon::parse($endDate)->format('d/m/Y'));
        $sheet->mergeCells('A2:J2');
        $sheet->getStyle('A2')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        $sheet->setCellValue('A3', 'Cabang: ' . $branchName);
        $sheet->mergeCells('A3:J3');
        $sheet->getStyle('A3')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);

        // Summary
        $row = 5;
        $sheet->setCellValue('A' . $row, 'RINGKASAN');
        $sheet->mergeCells('A' . $row . ':B' . $row);
        $sheet->getStyle('A' . $row)->getFont()->setBold(true)->setSize(12);
        $sheet->getStyle('A' . $row)->getFill()
            ->setFillType(Fill::FILL_SOLID)
            ->getStartColor()->setARGB('FFE0E0E0');
        $row++;
        $sheet->setCellValue('A' . $row, 'Total Pendapatan:');
        $sheet->setCellValue('B' . $row, 'Rp ' . number_format($totalRevenue, 0, ',', '.'));
        $sheet->getStyle('A' . $row . ':B' . $row)->getFont()->setBold(true);
        $row++;
        $sheet->setCellValue('A' . $row, 'Biaya Jasa Service:');
        $sheet->setCellValue('B' . $row, 'Rp ' . number_format($totalServiceFee, 0, ',', '.'));
        $row++;
        $sheet->setCellValue('A' . $row, 'Biaya Spare Part:');
        $sheet->setCellValue('B' . $row, 'Rp ' . number_format($totalPartsCost, 0, ',', '.'));
        $row++;
        $sheet->setCellValue('A' . $row, 'Sisa Pembayaran:');
        $sheet->setCellValue('B' . $row, 'Rp ' . number_format($totalRemaining, 0, ',', '.'));
        $row++;
        $sheet->setCellValue('A' . $row, 'Down Payment (DP):');
        $sheet->setCellValue('B' . $row, 'Rp ' . number_format($totalDownPayment, 0, ',', '.'));
        $row++;
        $sheet->setCellValue('A' . $row, 'Rata-rata DP:');
        $sheet->setCellValue('B' . $row, ($totalRevenue > 0 ? number_format(($totalDownPayment / $totalRevenue) * 100, 1) : 0) . '%');
        $row++;
        $sheet->setCellValue('A' . $row, 'Rata-rata Nilai per Service:');
        $sheet->setCellValue('B' . $row, 'Rp ' . ($services->count() > 0 && $totalRevenue > 0 ? number_format($totalRevenue / $services->count(), 0, ',', '.') : 0));

        // Pendapatan Per Cabang
        $row += 2;
        $sheet->setCellValue('A' . $row, 'PENDAPATAN PER CABANG');
        $sheet->mergeCells('A' . $row . ':F' . $row);
        $sheet->getStyle('A' . $row)->getFont()->setBold(true)->setSize(12);
        $sheet->getStyle('A' . $row)->getFill()
            ->setFillType(Fill::FILL_SOLID)
            ->getStartColor()->setARGB('FFE0E0E0');
        $row++;

        // Branch Table Header
        $branchHeaders = ['Cabang', 'Jumlah Service', 'Biaya Jasa', 'Biaya Spare Part', 'Total Pendapatan', 'Persentase'];
        $col = 'A';
        foreach ($branchHeaders as $header) {
            $sheet->setCellValue($col . $row, $header);
            $sheet->getStyle($col . $row)->getFont()->setBold(true);
            $sheet->getStyle($col . $row)->getFill()
                ->setFillType(Fill::FILL_SOLID)
                ->getStartColor()->setARGB('FF4CAF50');
            $sheet->getStyle($col . $row)->getFont()->getColor()->setARGB('FFFFFFFF');
            $col++;
        }

        // Branch Data
        $row++;
        foreach ($branchRevenue as $revenue) {
            $sheet->setCellValue('A' . $row, $revenue['branch_name']);
            $sheet->setCellValue('B' . $row, $revenue['total_services']);
            $sheet->setCellValue('C' . $row, 'Rp ' . number_format($revenue['total_service_fee'], 0, ',', '.'));
            $sheet->setCellValue('D' . $row, 'Rp ' . number_format($revenue['total_parts_cost'], 0, ',', '.'));
            $sheet->setCellValue('E' . $row, 'Rp ' . number_format($revenue['total_revenue'], 0, ',', '.'));
            $percentage = $totalRevenue > 0 ? number_format(($revenue['total_revenue'] / $totalRevenue) * 100, 1) : 0;
            $sheet->setCellValue('F' . $row, $percentage . '%');
            $row++;
        }

        // Branch Total
        $sheet->setCellValue('A' . $row, 'TOTAL');
        $sheet->setCellValue('B' . $row, $branchRevenue->sum('total_services'));
        $sheet->setCellValue('C' . $row, 'Rp ' . number_format($totalServiceFee, 0, ',', '.'));
        $sheet->setCellValue('D' . $row, 'Rp ' . number_format($totalPartsCost, 0, ',', '.'));
        $sheet->setCellValue('E' . $row, 'Rp ' . number_format($totalRevenue, 0, ',', '.'));
        $sheet->setCellValue('F' . $row, '100%');
        $sheet->getStyle('A' . $row . ':F' . $row)->getFont()->setBold(true);
        $sheet->getStyle('A' . $row . ':F' . $row)->getFill()
            ->setFillType(Fill::FILL_SOLID)
            ->getStartColor()->setARGB('FFF0F0F0');

        // Detail Transaksi Service
        $row += 2;
        $sheet->setCellValue('A' . $row, 'DETAIL TRANSAKSI SERVICE');
        $sheet->mergeCells('A' . $row . ':J' . $row);
        $sheet->getStyle('A' . $row)->getFont()->setBold(true)->setSize(12);
        $sheet->getStyle('A' . $row)->getFill()
            ->setFillType(Fill::FILL_SOLID)
            ->getStartColor()->setARGB('FFE0E0E0');
        $row++;

        // Detail Table Header
        $headers = ['No. Service', 'Tanggal', 'Customer', 'Device', 'Cabang', 'Biaya Jasa', 'Spare Part', 'Total', 'DP', 'Sisa'];
        $col = 'A';
        foreach ($headers as $header) {
            $sheet->setCellValue($col . $row, $header);
            $sheet->getStyle($col . $row)->getFont()->setBold(true);
            $sheet->getStyle($col . $row)->getFill()
                ->setFillType(Fill::FILL_SOLID)
                ->getStartColor()->setARGB('FF4CAF50');
            $sheet->getStyle($col . $row)->getFont()->getColor()->setARGB('FFFFFFFF');
            $col++;
        }

        // Detail Data
        $row++;
        $startDetailRow = $row;
        foreach ($services as $service) {
            $sheet->setCellValue('A' . $row, $service->service_number);
            $sheet->setCellValue('B' . $row, $service->service_date->format('d/m/Y'));
            $sheet->setCellValue('C' . $row, $service->customer->name);
            $sheet->setCellValue('D' . $row, $service->device_brand . ' ' . $service->device_model);
            $sheet->setCellValue('E' . $row, $service->branch->name);
            $sheet->setCellValue('F' . $row, 'Rp ' . number_format($service->service_fee, 0, ',', '.'));
            $sheet->setCellValue('G' . $row, 'Rp ' . number_format($service->parts_cost, 0, ',', '.'));
            $sheet->setCellValue('H' . $row, 'Rp ' . number_format($service->total_cost, 0, ',', '.'));
            $sheet->setCellValue('I' . $row, 'Rp ' . number_format($service->down_payment, 0, ',', '.'));
            $sheet->setCellValue('J' . $row, 'Rp ' . number_format($service->remaining_payment, 0, ',', '.'));
            $row++;
        }

        // Detail Total
        $sheet->setCellValue('A' . $row, 'TOTAL:');
        $sheet->mergeCells('A' . $row . ':E' . $row);
        $sheet->getStyle('A' . $row)->getAlignment()->setHorizontal(Alignment::HORIZONTAL_RIGHT);
        $sheet->setCellValue('F' . $row, 'Rp ' . number_format($totalServiceFee, 0, ',', '.'));
        $sheet->setCellValue('G' . $row, 'Rp ' . number_format($totalPartsCost, 0, ',', '.'));
        $sheet->setCellValue('H' . $row, 'Rp ' . number_format($totalRevenue, 0, ',', '.'));
        $sheet->setCellValue('I' . $row, 'Rp ' . number_format($totalDownPayment, 0, ',', '.'));
        $sheet->setCellValue('J' . $row, 'Rp ' . number_format($totalRemaining, 0, ',', '.'));
        $sheet->getStyle('A' . $row . ':J' . $row)->getFont()->setBold(true);
        $sheet->getStyle('A' . $row . ':J' . $row)->getFill()
            ->setFillType(Fill::FILL_SOLID)
            ->getStartColor()->setARGB('FFF0F0F0');

        // Borders for detail table
        $sheet->getStyle('A' . $startDetailRow . ':J' . $row)->getBorders()->getAllBorders()
            ->setBorderStyle(Border::BORDER_THIN);

        // Auto size columns
        foreach (range('A', 'J') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }

        $writer = new Xlsx($spreadsheet);
        $filename = 'laporan-pendapatan-service-' . $startDate . '-' . $endDate . '.xlsx';

        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }
}
