<?php

namespace App\Http\Controllers\Master;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Brand;

class BrandController extends Controller
{
    public function __construct()
    {
        $this->middleware(['permission:brands.index'])->only('index');
        $this->middleware(['permission:brands.create'])->only('save');
        $this->middleware(['permission:brands.edit'])->only('update');
        $this->middleware(['permission:brands.delete'])->only('delete');
    }

    public function index()
    {
        $title = "brand";
        $brands = Brand::orderBy('id', 'desc')->get();

        return view('master.brand', compact('title','brands'));
    }

    public function save(Request $request)
    {
        $this->validate($request, [
            'name' => 'required|unique:brands'
        ]);

        $brand = Brand::create([
            'name' => $request->input('name')
        ]);

        if($brand){
            //redirect dengan pesan sukses
            return redirect()->route('brand')->with(['success' => 'Data Berhasil Disimpan!']);
        }else{
            //redirect dengan pesan error
            return redirect()->route('brand')->with(['error' => 'Data Gagal Disimpan!']);
        }
    }

    public function update(Request $request, Brand $brand)
    {
        $this->validate($request, [
            'name' => 'required|unique:brands,name,'.$brand->id
        ]);

        $brand = Brand::findOrFail($brand->id);
        $brand->update([
            'name' => $request->input('name')
        ]);

        if($brand){
            //redirect dengan pesan sukses
            return redirect()->route('brand')->with(['success' => 'Data Berhasil Disimpan!']);
        }else{
            //redirect dengan pesan error
            return redirect()->route('brand')->with(['error' => 'Data Gagal Disimpan!']);
        }
    }

    public function delete(Brand $brand)
    {
        $brand->delete();

        return redirect()->route('brand')->with('success', 'Data Berhasil Dihapus');
    }

    /**
     * Download template Excel untuk import merek
     */
    public function downloadTemplate()
    {
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        // Set header
        $sheet->setCellValue('A1', 'Nama Merek');

        // Style header
        $sheet->getStyle('A1')->getFont()->setBold(true);
        $sheet->getStyle('A1')->getFill()
            ->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID)
            ->getStartColor()->setARGB('FFCCCCCC');

        // Set column width
        $sheet->getColumnDimension('A')->setWidth(40);

        // Add example data
        $sheet->setCellValue('A2', 'Contoh Merek 1');
        $sheet->setCellValue('A3', 'Contoh Merek 2');

        // Create Excel file
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $fileName = 'template_import_merek.xlsx';

        // Set headers for download
        header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
        header('Content-Disposition: attachment;filename="' . $fileName . '"');
        header('Cache-Control: max-age=0');

        $writer->save('php://output');
        exit;
    }

    /**
     * Import merek dari Excel
     */
    public function importExcel(Request $request)
    {
        $this->validate($request, [
            'excel_file' => 'required|mimes:xlsx,xls|max:2048',
        ]);

        try {
            $file = $request->file('excel_file');
            $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($file->getPathname());
            $sheet = $spreadsheet->getActiveSheet();
            $rows = $sheet->toArray();

            // Cek apakah file kosong
            if (count($rows) < 2) {
                return redirect()->back()->with('error', 'File Excel kosong atau format tidak sesuai!');
            }

            $successCount = 0;
            $errorCount = 0;
            $errors = [];

            // Mulai dari baris 2 (index 1) - setelah header
            for ($i = 1; $i < count($rows); $i++) {
                $row = $rows[$i];

                // Skip jika baris kosong
                if (empty($row[0])) {
                    continue;
                }

                $rowNumber = $i + 1;
                $brandName = isset($row[0]) ? trim($row[0]) : '';

                // Validasi data wajib
                if (empty($brandName)) {
                    $errors[] = "Baris {$rowNumber}: Nama merek tidak boleh kosong";
                    $errorCount++;
                    continue;
                }

                // Cek apakah merek sudah ada
                $existingBrand = Brand::where('name', $brandName)->first();

                if ($existingBrand) {
                    // Skip jika sudah ada
                    $errors[] = "Baris {$rowNumber}: Merek '{$brandName}' sudah ada";
                    $errorCount++;
                } else {
                    // Buat merek baru
                    Brand::create([
                        'name' => $brandName
                    ]);
                    $successCount++;
                }
            }

            $message = "Import selesai! Berhasil: {$successCount}, Gagal: {$errorCount}";

            if ($errorCount > 0) {
                $message .= "\n\nDetail error:\n" . implode("\n", array_slice($errors, 0, 10));
                if (count($errors) > 10) {
                    $message .= "\n... dan " . (count($errors) - 10) . " error lainnya";
                }
            }

            if ($successCount > 0) {
                return redirect()->route('brand')->with('success', $message);
            } else {
                return redirect()->route('brand')->with('error', $message);
            }

        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Gagal import Excel: ' . $e->getMessage());
        }
    }

}
